#ifndef NIXNET_ERRNO_HPP_
#define NIXNET_ERRNO_HPP_

#include <errno.h>      // for underlying errno
#include <expected>     // for std::expected
#include <iostream>     // for std::cerr
#include <string>       // for std::string
#include <string_view>  // for std::string_view
#include <type_traits>  // for std::remove_reference

namespace nixnet {

// possible errors that can be encountered
// For a detailed list of what errors are possible from
// each function, look at the documentation
enum class errno_t : std::remove_reference<decltype(errno)>::type {
  SUCCESS = 0,                   // ERRNO is zero to indicate there was no issue
  TOOBIG = E2BIG,                // Argument list too long (POSIX.1-2001).
  ACCES = EACCES,                // Permission denied (POSIX.1-2001)
  ADDRINUSE = EADDRINUSE,        // Address already in use (POSIX.1-2001).
  ADDRNOTAVAIL = EADDRNOTAVAIL,  // Address not available (POSIX.1-2001).
  AFNOSUPPORT = EAFNOSUPPORT,    // Address family not supported (POSIX.1-2001).
  AGAIN = EAGAIN,  // Resource temporarily unavailable (may be the same value as
                   // EWOULDBLOCK) (POSIX.1-2001).
  ALREADY = EALREADY,          // Connection already in progress (POSIX.1-2001).
  BADE = EBADE,                // Invalid exchange.
  BADF = EBADF,                // Bad file descriptor (POSIX.1-2001).
  BADFD = EBADFD,              // File descriptor in bad state.
  BADMSG = EBADMSG,            // Bad message (POSIX.1-2001).
  BADR = EBADR,                // Invalid request descriptor.
  BADRQC = EBADRQC,            // Invalid request code.
  BADSLT = EBADSLT,            // Invalid slot.
  BUSY = EBUSY,                // Device or resource busy (POSIX.1-2001).
  CANCELED = ECANCELED,        // Operation canceled (POSIX.1-2001).
  CHILD = ECHILD,              // No child processes (POSIX.1-2001).
  CHRNG = ECHRNG,              // Channel number out of range.
  COMM = ECOMM,                // Communication error on send.
  CONNABORTED = ECONNABORTED,  // Connection aborted (POSIX.1-2001).
  CONNREFUSED = ECONNREFUSED,  // Connection refused (POSIX.1-2001).
  CONNRESET = ECONNRESET,      // Connection reset (POSIX.1-2001).
  DEADLK = EDEADLK,            // Resource deadlock avoided (POSIX.1-2001).
  DESTADDRREQ = EDESTADDRREQ,  // Destination address required (POSIX.1-2001).
  DOM = EDOM,  // Mathematics argument out of domain of function (POSIX.1, C99).
  DQUOT = EDQUOT,              // Disk quota exceeded (POSIX.1-2001).
  EXIST = EEXIST,              // File exists (POSIX.1-2001).
  FAULT = EFAULT,              // Bad address (POSIX.1-2001).
  FBIG = EFBIG,                // File too large (POSIX.1-2001).
  HOSTDOWN = EHOSTDOWN,        // Host is down.
  HOSTUNREACH = EHOSTUNREACH,  // Host is unreachable (POSIX.1-2001).
  HWPOISON = EHWPOISON,        // Memory page has hardware error.
  IDRM = EIDRM,                // Identifier removed (POSIX.1-2001).
  ILSEQ = EILSEQ,  // Invalid or incomplete multibyte or wide character
                   // (POSIX.1, C99).
  INPROGRESS = EINPROGRESS,  // Operation in progress (POSIX.1-2001).
  INTR = EINTR,      // Interrupted function call (POSIX.1-2001); see signal(7).
  INVAL = EINVAL,    // Invalid argument (POSIX.1-2001).
  IO = EIO,          // Input/output error (POSIX.1-2001).
  ISCONN = EISCONN,  // Socket is connected (POSIX.1-2001).
  ISDIR = EISDIR,    // Is a directory (POSIX.1-2001).
  ISNAM = EISNAM,    // Is a named type file.
  KEYEXPIRED = EKEYEXPIRED,    // Key has expired.
  KEYREJECTED = EKEYREJECTED,  // Key has been revoked.
  L2HLT = EL2HLT,              // Level 2 halted.
  L2NSYNC = EL2NSYNC,          // Level 2 not synchronized
  L3HLT = EL3HLT,              // Level 3 halted.
  L3RST = EL3RST,              // Level 3 reset.
  LIBACC = ELIBACC,            // Cannot access a needed shared library.
  LIBBAD = ELIBBAD,            // Accessing a corrupted shared library.
  LIBMAX = ELIBMAX,    // Attempting to link in too many shared libraries.
  LIBSCN = ELIBSCN,    // .lib section in a.out corrupted
  LIBEXEC = ELIBEXEC,  // Cannot exec a shared library directly.
#ifdef ELNRANGE
  LNRANGE = ELNRANGE,  // Link number out of range.
#endif
  LOOP = ELOOP,  // Too many levels of symbolic links (POSIX.1-2001).
  MEDIUMTYPE = EMEDIUMTYPE,  // Wrong medium type.
  MFILE = EMFILE,  // Too many open files (POSIX.1-2001).  Commonly caused by
                   // exceeding the RLIMIT_NOFILE resource limit described in
                   // getrlimit(2).  Can also be caused by exceeding the limit
                   // specified  in /proc/sys/fs/nr_open.
  MLINK = EMLINK,  // Too many links (POSIX.1-2001).
  MSGSIZE = EMSGSIZE,          // Message too long (POSIX.1-2001).
  MULTIHOP = EMULTIHOP,        // Multihop attempted (POSIX.1-2001).
  NAMETOOLONG = ENAMETOOLONG,  // Filename too long (POSIX.1-2001).
  NETDOWN = ENETDOWN,          // Network is down (POSIX.1-2001).
  NETRESET = ENETRESET,        // Connection aborted by network (POSIX.1-2001).
  NETUNREACH = ENETUNREACH,    // Network unreachable (POSIX.1-2001).
  NFILE = ENFILE,  // Too many open files in system (POSIX.1-2001).  On Linux,
                   // this is probably a result of encountering the
                   // /proc/sys/fs/file-max limit (see proc(5)).
  NOANO = ENOANO,  // No anode.
  NOBUFS =
      ENOBUFS,  // No buffer space available (POSIX.1 (XSI STREAMS option)).
  NODATA = ENODATA,  // No message is available on the STREAM head read queue
                     // (POSIX.1-2001).
  NODEV = ENODEV,    // No such device (POSIX.1-2001).
  NOENT = ENOENT,    // No such file or directory (POSIX.1-2001).
  NOEXEC = ENOEXEC,  // Exec format error (POSIX.1-2001).
  NOKEY = ENOKEY,    // Required key not available.
  NOLCK = ENOLCK,    // No locks available (POSIX.1-2001).
  NOLINK = ENOLINK,  // Link has been severed (POSIX.1-2001).
  NOMEDIUM = ENOMEDIUM,  // No medium found.
  NOMEM = ENOMEM,  // Not enough space/cannot allocate memory (POSIX.1-2001).
  NOMSG = ENOMSG,  // No message of the desired type (POSIX.1-2001).
  NONET = ENONET,  // Machine is not on the network.
  NOPKG = ENOPKG,  // Package not installed.
  NOPROTOOPT = ENOPROTOOPT,  // Protocol not available (POSIX.1-2001).
  NOSPC = ENOSPC,            // No space left on device (POSIX.1-2001).
  NOSR = ENOSR,          // No STREAM resources (POSIX.1 (XSI STREAMS option)).
  NOSTR = ENOSTR,        // Not a STREAM (POSIX.1 (XSI STREAMS option)).
  NOSYS = ENOSYS,        // Function not implemented (POSIX.1-2001).
  NOTBLK = ENOTBLK,      // Block device required.
  NOTCONN = ENOTCONN,    // The socket is not connected (POSIX.1-2001).
  NOTDIR = ENOTDIR,      // Not a directory (POSIX.1-2001).
  NOTEMPTY = ENOTEMPTY,  // Directory not empty (POSIX.1-2001).
  NOTRECOVERABLE = ENOTRECOVERABLE,  // State not recoverable (POSIX.1-2008).
  NOTSOCK = ENOTSOCK,                // Not a socket (POSIX.1-2001).
  NOTSUP = ENOTSUP,                  // Operation not supported (POSIX.1-2001).
  NOTTY = ENOTTY,      // Inappropriate I/O control operation (POSIX.1-2001).
  NOTUNIQ = ENOTUNIQ,  // Name not unique on network.
  NXIO = ENXIO,        // No such device or address (POSIX.1-2001).
  OPNOTSUPP = EOPNOTSUPP,  // Operation not supported on socket (POSIX.1-2001).
  OVERFLOW =
      EOVERFLOW,  // Value too large to be stored in data type (POSIX.1-2001).
  OWNERDEAD = EOWNERDEAD,            // Owner died (POSIX.1-2008).
  PERM = EPERM,                      // Operation not permitted (POSIX.1-2001).
  PFNOSUPPORT = EPFNOSUPPORT,        // Protocol family not supported.
  PIPE = EPIPE,                      // Broken pipe (POSIX.1-2001).
  PROTO = EPROTO,                    // Protocol error (POSIX.1-2001).
  PROTONOSUPPORT = EPROTONOSUPPORT,  // Protocol not supported (POSIX.1-2001).
  PROTOTYPE = EPROTOTYPE,  // Protocol wrong type for socket (POSIX.1-2001).
  RANGE = ERANGE,          // Result too large (POSIX.1, C99).
  REMCHG = EREMCHG,        // Remote address changed.
  REMOTE = EREMOTE,        // Object is remote.
  REMOTEIO = EREMOTEIO,    // Remote I/O error.
  RESTART = ERESTART,      // Interrupted system call should be restarted.
  RFKILL = ERFKILL,        // Operation not possible due to RF-kill.
  ROFS = EROFS,            // Read-only filesystem (POSIX.1-2001).
  SHUTDOWN = ESHUTDOWN,    // Cannot send after transport endpoint shutdown.
  SPIPE = ESPIPE,          // Invalid seek (POSIX.1-2001).
  SOCKTNOSUPPORT = ESOCKTNOSUPPORT,  // Socket type not supported.
  SRCH = ESRCH,                      // No such process (POSIX.1-2001).
  STALE = ESTALE,  // Stale file handle (POSIX.1-2001). This error can occur for
                   // NFS and for other filesystems.
  STRPIPE = ESTRPIPE,  // Streams pipe error.
  TIME = ETIME,  // Timer expired (POSIX.1 (XSI STREAMS option)). (POSIX.1 says
                 // "STREAM ioctl(2) timeout".)
  TIMEDOUT = ETIMEDOUT,        // Connection timed out (POSIX.1-2001).
  TOOMANYREFS = ETOOMANYREFS,  //
  TXTBSY = ETXTBSY,            // Text file busy (POSIX.1-2001).
  UCLEAN = EUCLEAN,            // Structure needs cleaning.
  UNATCH = EUNATCH,            // Protocol driver not attached.
  USERS = EUSERS,              // Too many users.
  WOULDBLOCK = EWOULDBLOCK,    // Operation would block (may be same value as
                               // EAGAIN) (POSIX.1-2001).
  XDEV = EXDEV,                // Improper link (POSIX.1-2001).
  XFULL = EXFULL,              // Exchange full.
};

// This is just an object that derives from std::expected so that
// I can add a function that rust has for their simialr type. Namely
// - result::expect()
template <typename T>
class result : public std::expected<T, errno_t> {
 public:
  // constructors
  using std::expected<T, errno_t>::expected;
  // observers
  using std::expected<T, errno_t>::operator->;
  using std::expected<T, errno_t>::operator*;
  using std::expected<T, errno_t>::operator bool;
  using std::expected<T, errno_t>::has_value;
  using std::expected<T, errno_t>::value;
  using std::expected<T, errno_t>::error;

  // new function
  //
  // T& result<T>::expect(std::string_view message)
  //
  // retuns the expected value inside of this, iff this contains an expected
  // value and not an error value.
  //
  // Iff this contains an error, instead the supplied message is printed to
  // cerr, along with a message describing the error code, and the
  // std::except::bad_access exception is thrown

  // called on a result&
  constexpr T& expect(std::string_view msg) & {
    if (!this->has_value()) {
      std::cerr << msg << ": " << str_error(this->error()) << '\n';
    }
    return this->value();
  }

  // called on a const result&
  constexpr const T& expect(std::string_view msg) const& {
    if (!this->has_value()) {
      std::cerr << msg << ": " << str_error(this->error()) << '\n';
    }
    return this->value();
  }

  // called on a result&&
  constexpr T&& expect(std::string_view msg) && {
    if (!this->has_value()) {
      std::cerr << msg << ": " << str_error(this->error()) << '\n';
    }
    return std::move(this->value());
  }

  // called on a const result&&
  constexpr const T&& expect(std::string_view msg) const&& {
    if (!this->has_value()) {
      std::cerr << msg << ": " << str_error(this->error()) << '\n';
    }
    return std::move(this->value());
  }
};

/* Given an errno_t error code, returns a std::string
 * containing a human readablle description of the
 * error code.
 */
[[nodiscard]] std::string str_error(errno_t err);

};  // namespace nixnet

#endif  // NIXNET_ERRNO_HPP_
