#ifndef NIXNET_UDPSOCKET_HPP_
#define NIXNET_UDPSOCKET_HPP_

#include <chrono>
#include <cstddef>
#include <span>
#include "./SocketAddr.hpp"
#include "./errno.hpp"

namespace nixnet {

// This class represents a UdpSocket
// UDP is an unordered, unreliable protocol for sending
// datagrams
//
// Once created created via either
// - UdpSocket::bind
// or
// - UdpSocket::unspified_local_addr
//
// the socket then can be used to send or receive data
//
// Note that this implementation checks the MTU for you
// and will return an error on send() or sendto() if you
// try to write too many bytes
//
// !!!!! IMPORTANT !!!!!!
// Since we are using this in an educational context and are often
// sending data from a socket on one computer to another socket
// on the same computer, MTU is a bit different.
// MTU (Maximum Transmission Unit) for sneding data on the same computer is
// something like 65,507 bytes. MTU can be a lot slower and so to
// demonstrate that (especially for one of the HWs), it is artificially set to
// 68 or 1280 (for IPv4 and IPv6 respectively). If you want to use these
// in a "real" context, then you need to delete the marked lines of code in
// the implementation of send() and sendto() to avoid this issue.
class UdpSocket {
 public:
  // Creates a UDP socket that has the given local address
  // and port number.
  //
  // calls the underlying POSIX system calls:
  // socket() and bind() to do this.
  //
  // On success returns a new UdpSocket bound to the specified
  // port and local address.
  //
  // To avoid too much repitition, look at the man pages for
  // socket() and bind() to see the possible error codes that can be returned.
  static result<UdpSocket> bind(const SocketAddr& addr);

  // Creates a UDP socket that has no specified
  // the local address or port number.
  // An IP Address and a Port will be allocated by
  // the OS when this socket sends data. Most likely the port
  // chosen will be one of the "ephemeral" or "temporary" ports.
  //
  //
  // calls the underlying POSIX system call:
  // socket() to do this.
  //
  // On success returns a new UdpSocket ready for you
  // to send data
  //
  // To avoid too much repitition, look at the man pages for
  // socket() and bind() to see the possible error codes that can be returned.
  static result<UdpSocket> unspecified_local_addr(sa_family_t family);

  // delete copying
  UdpSocket(const UdpSocket& other) = delete;
  UdpSocket& operator=(const UdpSocket& other) = delete;

  // only support moving
  UdpSocket(UdpSocket&& other);
  UdpSocket& operator=(UdpSocket&& other);

  // closes the socket
  ~UdpSocket();

  // closes the socket
  // afterwards all other functions will fail when called on this
  // unless this socket is used as the destination as part of
  // a move assignmen.
  void close();

  // sends the specified bytes to the specified address as a datagram.
  // returns the number of bytes successfully sent.
  //
  // Is a wrapper around the sendto() POSIX system call
  //
  // One possible error that can be returned is errno_t::MSGSIZE
  // if you try to write too many bytes at once.
  //
  // To avoid too much repitition, look at the man page for
  // sendto() to see the other error codes that can be returned.
  result<size_t> sendto(std::span<const std::byte> bytes,
                        const SocketAddr& addr) const;

  // Receives a single datagram message from the socket.
  // Returns the number of bytes read and the address thaat the datagram was
  // sent from
  //
  // Note that if the passed in span is not big enough to hold all of the
  // bytes in the datagram message read in, then the excess bytes may be
  // discarded.
  //
  // Is a wrapper around the recvfrom() POSIX system call
  //
  // To avoid too much repitition, look at the man page for
  // recvfrom() to see the other error codes that can be returned.
  result<std::pair<size_t, SocketAddr>> recvfrom(
      std::span<std::byte> bytes) const;

  // Connects this UdpSocket to the remote address.
  // This lets you use the two functions below without having to specify the
  // address we are sending/receiving datagrams to/from.
  //
  // This function also makes it so that other datagrams sent by other
  // addresses will be ignored.
  //
  // Is a wrapper around the connect() POSIX system call
  //
  // To avoid too much repitition, look at the man page for
  // connect() to see the other error codes that can be returned.
  errno_t connect(const SocketAddr& addr);

  // sends the specified bytes to the address this socket is connected to..
  // returns the number of bytes successfully sent.
  //
  // Is a wrapper around the send() POSIX system call
  //
  // Fails and returns errno_t::NOTCONN if the socket is
  // not connected to an address (otherwise send() doesn't know
  // where to send the data)
  //
  // One possible error that can be returned is errno_t::MSGSIZE
  // if you try to write too many bytes at once.
  //
  // To avoid too much repitition, look at the man page for
  // send() to see the other error codes that can be returned.
  result<size_t> send(std::span<std::byte> bytes) const;

  // the same as recvfrom but it doesn't return the
  // address of the sender of the datagram read in.
  result<size_t> recv(std::span<std::byte> bytes) const;

  // Sets the read timeout to the specified number of microseconds
  // If UdpSocket calls recv or recvfrom and the specified number of
  // microseconds pass without receiving data, then those function will
  // return error with either errno_t::WOULDBLOCK or errno_t::TimedOut
  // The exact value returend is operaating system defined.
  //
  // If 0 is passed in, then the timeout is turned off, and the functions
  // recv and recvfrom will instead block indefinitely or until something is
  // read in.
  //
  // returns errno_t::SUCCESS on success.
  //
  // wrapper around setsockopt SO_RCVTIMEO
  errno_t set_read_timeout(std::chrono::microseconds time);

  // Same as above but for the writing instead of reading
  //
  // wrapper around setsockopt SO_SNDTIMEO
  errno_t set_write_timeout(std::chrono::microseconds time);

  // returns how long the timeout is for reading or writing
  //
  // wrapper around getsockopt for SO_RCVTIMO and SO_SNDTIMEO
  //
  // returns errno_t::SUCCESS on success;
  result<std::chrono::microseconds> read_timeout() const;
  result<std::chrono::microseconds> write_timeout() const;

  // returns the underlying file descriptor (socket) that this
  // UdpSocket is a wrapper around. Returns -1 if this socket
  // was closed previously.
  int raw_fd();

  // Returns the SockAddr this socket was
  // created from with bind or assigned by
  // the os if the unspecified_local_addr function was used.
  //
  // returns error if this socket is closed
  result<SocketAddr> local_addr() const;

  // Returns the SockAddr of the peer
  // that this socket is connected to.
  //
  // returns error if this socket is closed or if
  // the socket is not connected to a specific peer
  result<SocketAddr> peer_addr() const;

  // returns the (fake) Maximum Transmission Unit
  // this is the number of bytes you can send via send or sendto
  // before it fails due to being unable to fit all the data into
  // a signle packet.
  size_t get_mtu() const;

 private:
  UdpSocket() : m_fd(-1){};
  int m_fd;
};

};  // namespace nixnet

#endif  // NIXNET_UDPSOCKET_HPP_
