#ifndef NIXNET_TCPSTREAM_HPP_
#define NIXNET_TCPSTREAM_HPP_

#include <chrono>
#include <cstddef>
#include <span>
#include "./SocketAddr.hpp"
#include "./errno.hpp"

namespace nixnet {

// This class represents a TcpSocket stream
// TCP provides a ordered, reliable byte stream
// abstraaction for sending bytes over the netowkr.
//
// Once created created via either
// - TcpStream::connect
// or
// - returned by TcpListner::accept
//
// the socket then can be used to send or receive data
class TcpStream {
 public:
  // Creates a new TCP socket that is connected to the remote host
  // specified by the passed in address.
  //
  // calls the underlying POSIX system calls:
  // socket() and connect() to do this.
  //
  // On success returns a new TcpStream connected to the specified
  // remote host. A local address and port will be automatically
  // be allocated by the operating system.
  //
  // To avoid too much repitition, look at the man pages for socket()
  // and connect() to see the possible error codes that can be returned.
  static result<TcpStream> connect(const SocketAddr& addr);

  // delete copying
  TcpStream(const TcpStream& other) = delete;
  TcpStream& operator=(const TcpStream& other) = delete;

  // only support moving
  TcpStream(TcpStream&& other);
  TcpStream& operator=(TcpStream&& other);

  // closes the socket
  ~TcpStream();

  // closes the socket
  // afterwards all other functions will fail when called on this
  // unless this socket is used as the destination as part of
  // a move assignmen.
  void close();

  // sends the specified bytes to the address this
  // stream is connected to. Bytes are sent via TCP
  // which means they are sent reliably and in order.
  //
  // Returns the number of bytes successfully sent
  // or an error code on error.
  //
  // Is a wrapper around the send() POSIX system call
  //
  // To avoid too much repitition, look at the man page for
  // recvv() to see the other error codes that can be returned.
  result<size_t> send(std::span<const std::byte> bytes) const;

  // receives up to the length of the specifeid span
  // number of bytes from the TcpStream into the span.
  // Bytes that are read were sent from the the address this
  // stream is connected to. Bytes are received via TCP
  // which means they are sent reliably and in order.
  //
  // Returns the number of bytes successfully read
  // or an error code on error.
  //
  // Is a wrapper around the recv() POSIX system call
  //
  // To avoid too much repitition, look at the man page for
  // recv() to see the other error codes that can be returned.
  result<size_t> recv(std::span<std::byte> bytes) const;

  // Sets the read timeout to the specified number of microseconds
  // If TcpSTream calls recv and the specified number of
  // microseconds pass without receiving data, then those function will
  // return error with either errno_t::WOULDBLOCK or errno_t::TimedOut
  // The exact value returend is operaating system defined.
  //
  // If 0 is passed in, then the timeout is turned off, and the functions
  // recv and recvfrom will instead block indefinitely or until something is
  // read in.
  //
  // returns errno_t::SUCCESS on success.
  //
  // wrapper around setsockopt SO_RCVTIMEO
  errno_t set_read_timeout(std::chrono::microseconds time);

  // Same as above but for the writing instead of reading
  //
  // wrapper around setsockopt SO_SNDTIMEO
  errno_t set_write_timeout(std::chrono::microseconds time);

  // returns how long the timeout is for reading or writing
  //
  // wrapper around getsockopt for SO_RCVTIMO and SO_SNDTIMEO
  //
  // returns errno_t::SUCCESS on success;
  result<std::chrono::microseconds> read_timeout() const;
  result<std::chrono::microseconds> write_timeout() const;

  // returns the underlying file descriptor (socket) that this
  // TcpStream is a wrapper around. Returns -1 if this socket
  // was closed previously.
  int raw_fd();

  // Returns the SockAddr associated with this socket
  //
  // returns error if this socket is closed
  result<SocketAddr> local_addr() const;

  // Returns the SockAddr of the peer
  // that this socket is connected to.
  //
  // returns error if this socket is closed.
  result<SocketAddr> peer_addr() const;

  // so that TcpListener has access to private ctor
  friend class TcpListener;

 private:
  // default ctor to a closed socket
  TcpStream() : m_fd(-1){};

  int m_fd;
};

};  // namespace nixnet

#endif  // NIXNET_TCPSTREAM_HPP_
