#ifndef NIXNET_TCPLISTENER_HPP_
#define NIXNET_TCPLISTENER_HPP_

#include <chrono>
#include <cstddef>
#include <span>
#include "./SocketAddr.hpp"
#include "./TcpStream.hpp"
#include "./errno.hpp"

namespace nixnet {

// This class represents a TCP socket listener
// that is bound to a local address and port. The
// socket can then be used to listen for incoming TCP connections
// calling accept() will return one of these connections bound to a new
// TCPStream object.
class TcpListener {
 public:
  // Creates a new TcpListener which will be bound to the specified address.
  // The moment that the listener is constructed, clinets will be able to
  // connect but will be queued until the connection is accepted.
  //
  // If an address with 0 as the port number is passed in, then the OS assigns a
  // port to this listener. The port allocated can be queried via the
  // TcpListener::local_addr method.
  //
  // calls the underlying POSIX system calls:
  // socket(), bind(), and listen() to do this.
  //
  // To avoid too much repitition, look at the man pages for socket()
  // bind() and listen() to see the possible error codes that can be returned.
  static result<TcpListener> bind(const SocketAddr& addr);

  // delete copying
  TcpListener(const TcpListener& other) = delete;
  TcpListener& operator=(const TcpListener& other) = delete;

  // only support moving
  TcpListener(TcpListener&& other);
  TcpListener& operator=(TcpListener&& other);

  // Destructor
  ~TcpListener();

  // closes the socket
  // afterwards all other functions will fail when called on this
  // unless this socket is used as the destination as part of
  // a move assignmen.
  void close();

  // returns the underlying file descriptor (socket) that this
  // TcpListener is wrapped around. Returns -1 if this socket
  // was closed previously.
  int raw_fd();

  // Accepts a new incoming connection.
  // This function blocks until a TCP connection is established
  //
  // Returns a new TcpStream object that is the local endpoint
  // for the newly accepted TCP connection. the address of the
  // remote peer for this new connection is also returned.
  //
  // calls the underlying POSIX system call:
  // accept() to do this.
  //
  // To avoid too much repitition, look at the man pages for acppet()
  // to see the possible error codes that can be returned.
  result<std::pair<TcpStream, SocketAddr>> accept() const;

  // Returns the SockAddr associated with this socket
  //
  // returns error if this socket is closed
  result<SocketAddr> local_addr() const;

 private:
  // default ctor to a closed socket
  TcpListener() : m_fd(-1){};
  int m_fd;
};

};  // namespace nixnet

#endif  // NIXNET_TCPLISTENER_HPP_
