#include "./TcpListener.hpp"

using std::byte;
using std::pair;
using std::span;
using std::unexpected;
using std::chrono::duration_cast;
using std::chrono::floor;
using std::chrono::microseconds;
using std::chrono::seconds;

#include <unistd.h>

namespace nixnet {

result<TcpListener> TcpListener::bind(const SocketAddr& addr) {
  TcpListener result_sock;
  int fd = ::socket(addr.data()->sa_family, SOCK_STREAM, 0);
  if (fd < 0) {
    return unexpected(errno_t(errno));
  }
  result_sock.m_fd = fd;
  // Configure the socket; we're setting a socket "option."  In
  // particular, we set "SO_REUSEADDR", which tells the TCP stack
  // so make the port we bind to available again as soon as we
  // exit, rather than waiting for a few tens of seconds to recycle it.
  int optval = 1;
  ::setsockopt(fd, SOL_SOCKET, SO_REUSEADDR, &optval, sizeof(optval));
  int res = ::bind(fd, addr.data(), addr.data_size());
  if (res != 0) {
    return unexpected(errno_t(errno));
  }
  res = ::listen(fd, SOMAXCONN);
  if (res != 0) {
    return unexpected(errno_t(errno));
  }
  return result_sock;
}

TcpListener::TcpListener(TcpListener&& other) : m_fd(other.m_fd) {
  other.m_fd = -1;
}

TcpListener& TcpListener::operator=(TcpListener&& other) {
  if (&other != this) {
    if (m_fd >= 0) {
      ::close(m_fd);
    }
    m_fd = other.m_fd;
    other.m_fd = -1;
  }
  return *this;
}

TcpListener::~TcpListener() {
  if (m_fd >= 0) {
    ::close(m_fd);
  }
}

void TcpListener::close() {
  if (m_fd >= 0) {
    ::close(m_fd);
    m_fd = -1;
  }
}

result<std::pair<TcpStream, SocketAddr>> TcpListener::accept() const {
  struct sockaddr_storage caddr;
  socklen_t caddr_len = sizeof(caddr);
  int client_fd =
      ::accept(m_fd, reinterpret_cast<struct sockaddr*>(&caddr), &caddr_len);
  if (client_fd < 0) {
    return unexpected(errno_t(errno));
  }
  TcpStream client;
  client.m_fd = client_fd;
  auto addr = SocketAddr::from_csocket(reinterpret_cast<sockaddr*>(&caddr));
  if (!addr) {
    return unexpected(addr.error());
  }
  return std::make_pair(std::move(client), std::move(addr.value()));
}

int TcpListener::raw_fd() {
  return m_fd;
}

result<SocketAddr> TcpListener::local_addr() const {
  sockaddr_storage ss;
  socklen_t ss_size = sizeof(ss);
  int temp = ::getsockname(m_fd, reinterpret_cast<sockaddr*>(&ss), &ss_size);
  if (temp < 0) {
    return unexpected(errno_t(errno));
  }

  return SocketAddr::from_csocket(reinterpret_cast<sockaddr*>(&ss));
};

};  // namespace nixnet
