#ifndef NIXNET_SOCKETADDR_HPP_
#define NIXNET_SOCKETADDR_HPP_

#include <arpa/inet.h>
#include <cstdint>
#include <ostream>
#include <string_view>
#include "./errno.hpp"

namespace nixnet {

constexpr const char* V4_INADDRANY = "0.0.0.0";
constexpr const char* V4_LOCALHOST = "127.0.0.1";
constexpr const char* V4_BROADCAST = "255.255.255.255";

constexpr const char* V6_INADDRANY = "::";
constexpr const char* V6_LOCALHOST = "::1";

class SocketAddr {
 public:
  // Use one of these functions to "construct" a SocketAddr
  static result<SocketAddr> v4_from_str(std::string_view raw,
                                        uint16_t port_num);
  static result<SocketAddr> v4_from_str(std::string_view raw,
                                        std::string_view port_str);
  static result<SocketAddr> v6_from_str(std::string_view raw,
                                        uint16_t port_num);
  static result<SocketAddr> v6_from_str(std::string_view raw,
                                        std::string_view port_str);
  static result<SocketAddr> from_csocket(const sockaddr* addr);

  SocketAddr(const SocketAddr& other) = default;
  SocketAddr(SocketAddr&& other) = default;

  SocketAddr& operator=(const SocketAddr& other) = default;
  SocketAddr& operator=(SocketAddr&& other) = default;

  result<std::string> to_string();

  sockaddr* data();
  const sockaddr* data() const;
  constexpr size_t data_size() const { return sizeof(m_data); };

  bool is_ipv4() const;
  bool is_ipv6() const;

  void set_port(uint16_t port_num);
  uint16_t port() const;

 private:
  // real constructor is protected so that you have to use one of
  // the from functions
  SocketAddr() = default;
  sockaddr_storage m_data;

  // helper function
  friend result<SocketAddr> pton_helper(sa_family_t family,
                                        std::string_view raw,
                                        uint16_t port_num);
};

std::string to_string(const SocketAddr& value);

std::ostream& operator<<(std::ostream& out, const SocketAddr& value);

};  // namespace nixnet

#endif  // NIXNET_SOCKETADDR_HPP_
