from django.contrib.auth import authenticate, login, logout
from django.contrib.auth.decorators import login_required
from django.contrib.auth.models import User

from django.shortcuts import render, get_object_or_404
from django.http import HttpResponseRedirect
from django.urls import reverse
from django.views import generic
from .models import Question, Choice, Vote

# Create your views here.

# We can define generic views like so
class IndexView(generic.ListView):
    template_name = "index.html"
    context_object_name = "questions"

    def get_queryset(self):
        return Question.objects.order_by("-created_at")[:5]

class DetailView(generic.DetailView):
    model = Question
    template_name = "detail.html"

class ResultsView(generic.DetailView):
    model = Question
    template_name = "results.html"


# or use view functions like so

def index(request):
    questions = Question.objects.order_by("-start_time")[:5]

    return render(request, "index.html", context={
        "questions": questions,
    })

def question_detail(request, question_id):
    question = get_object_or_404(Question, id=question_id)

    return render(request, "detail.html", context={
        "question": question,
    })

def question_results(request, question_id):
    question = get_object_or_404(Question, id=question_id)

    return render(request, "results.html", context={
        "question": question,
    })

# Note the use of the decorator for user authentication
@login_required
def vote(request, question_id):
    question = get_object_or_404(Question, id=question_id)
    choice_id = request.POST['choice']
    choice = get_object_or_404(Choice, id=choice_id)

    try: 
        Vote.objects.create(
            choice=choice,
            user=request.user,
        )

        return render(request, "detail.html", context={
            "question": question,
            "message": "voted successfully!",
        })
    except Exception as e:
        print(e)
        return render(request, "detail.html", context={
            "question": question,
            "message": "vote failed!",
        })


### User Auth

def login_view(request):
    if request.method == "GET":
        return render(request, "login.html")

    try: 
        username = request.POST['username']
        password = request.POST['password']
        user = authenticate(request, username=username, password=password)

        if user is not None:
            login(request, user)
            return render(request, "login.html", context={
                "message": "successfully logged in!",
            })
    finally:
        return render(request, "login.html", context={
            "message": "invalid credentials!"
        })

def logout_view(request):
    logout(request)
    return HttpResponseRedirect(reverse("polls:homepage"))

def signup(request):
    if request.method == "GET":
        return render(request, "signup.html")

    try:
        fname = request.POST['fname']
        lname = request.POST['lname']
        username = request.POST['username']
        password = request.POST['password']

        User.objects.create_user(
            username=username,
            password=password,
            first_name=fname,
            last_name=lname,
        )
        return render(request, "signup.html", context={
            "message": "successfully created user!"
        })
    except Exception as e:
        return render(request, "signup.html", context={
            "message": f"something went wrong: {e}"
        })    
