from django.contrib.auth.models import User
from django.db import models
from django.db.models import Count, UniqueConstraint
from django.utils import timezone


# Create your models here.
class Question(models.Model):
    text = models.CharField(max_length=200)
    start_time = models.DateTimeField()
    end_time = models.DateTimeField()

    # Typically we have creation and update timestamps on database models.
    # These are easily implemented with the `auto_now_add` and `auto_now` parameters.
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    # We can override methods on the model, i.e. overriding __str__
    # makes it so that we can easily read it in the Django admin site.
    def __str__(self):
        return f"({self.id}) {self.text}"
    
    # custom methods can also work
    def is_active(self):
        return self.start_time <= timezone.now() <= self.end_time
    
    def get_choices_with_votes(self):
        return self.choice_set.annotate(vote_count=Count('vote'))

class Choice(models.Model):
    choice = models.CharField(max_length=200)
    question = models.ForeignKey(Question, on_delete=models.CASCADE)
    
    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)


class Vote(models.Model):
    choice = models.ForeignKey(Choice, on_delete=models.CASCADE)
    user = models.ForeignKey(User, on_delete=models.CASCADE)

    created_at = models.DateTimeField(auto_now_add=True)
    updated_at = models.DateTimeField(auto_now=True)

    # This meta class allows us to define extra database level things
    class Meta:
        # We use it to additionally define a *constraint*:
        # No user can vote for the same choice more than once
        constraints = [
            UniqueConstraint(
                fields=['choice', 'user'], 
                name='unique_user_vote'),
        ]
